/**
* \file: cairo-font-handling.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* Example for drawing graphics primitives with cairo.
* \component: graphics demos
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include "cairo-font-handling.h"

/**
 * @brief ft_cairo_user_data_key
 *
 * Key structure used for associating FT_Face with cairo_font_face user data.
 * See http://cairographics.org/manual/cairo-FreeType-Fonts.html#cairo-ft-font-face-create-for-ft-face
 * for details on why this is used and
 * http://cairographics.org/manual/cairo-Types.html#cairo-user-data-key-t for
 * why it does not matter to not have a different one for each font.
 */
static cairo_user_data_key_t ft_cairo_user_data_key;

/**
 * @brief Global FreeType library handle.
 */
static FT_Library ft;

/**
 * @brief cf_new create a new cairo font face from a truetype font file
 * @param font_file path to TTF file
 * @return NULL on failure, a valid cairo_font_face_t on success.
 *
 * Example function of setting up a cairo font from a TTF file. Also adds the
 * functionality recommended by the cairo documentation to auto-destroy the
 * FT_Face with the cairo font.
 */
cairo_font_face_t *cairo_font_from_ttf_file(const char *font_file)
{
    FT_Error result = 0;
    FT_Face face;
    cairo_status_t status = CAIRO_STATUS_SUCCESS;
    cairo_font_face_t *cf = NULL;

    /* Try to create the FreeType library handle on-the-fly */
    if (NULL == ft)
    {
        result = FT_Init_FreeType(&ft);
        if (0 != result)
        {
            fprintf(stderr, "cairo_font_from_ttf_file: Failed to initialize "
                            "FreeType library\n");
            return NULL;
        }
    }

    if (NULL == font_file)
    {
        fprintf(stderr, "cairo_font_from_ttf_file: font_file path was not given\n");
        return NULL;
    }


    /* Creating a cairo font from any font file supported by FreeType is pretty
     * straight forward.
     *
     * First, we have to create the FT_Face from the font file using FT_New_Face
     */
    result = FT_New_Face(ft, font_file, 0, &face);
    if (0 != result)
    {
        fprintf(stderr, "cairo_font_from_ttf_file: Failed to create face\n");
        return NULL;
    }

    /*
     * Then we can pass the font face to cairo_ft_font_face_create_for_ft_face()
     * to create a cairo_font_face_t which can be easily used during normal
     * cairo drawing activities.
     */
    cf = cairo_ft_font_face_create_for_ft_face(face, 0);
    if (cf == NULL)
    {
        fprintf(stderr, "cairo_font_from_ttf_file: Failed to create cairo font "
                        "from FT_Face\n");
        FT_Done_Face(face);

        return NULL;
    }

    /* The FT_Face needs to be valid for as long as the cairo_font_face_t is
     * alive. Since the lifetime of the cairo_font_face_t is not easily
     * determined from the outside, cairo documentation recommends to couple
     * the life-time of the FT_Face on the lifetime setting it as its user-data
     * which will be freed if the cairo_font_face_t is freed.
     *
     * See also http://cairographics.org/manual/cairo-FreeType-Fonts.html#cairo-ft-font-face-create-for-ft-face
     * for the reasoning.
     */
    status = cairo_font_face_set_user_data(cf,
                                           &ft_cairo_user_data_key,
                                           face,
                                           (cairo_destroy_func_t) FT_Done_Face);

    if (status != CAIRO_STATUS_SUCCESS)
    {
        cairo_font_face_destroy(cf);
        FT_Done_Face(face);

        return NULL;
    }

    return cf;
}

/**
 * @brief Example for drawing a non-complex but also non-ascii script.
 * @param cr is a cairo context to draw to
 * @param width of the screen
 * @param height of the screen
 *
 * This is an example of drawing text on-screen that contains more than ASCII.
 * All that is needed is a font file that contains the glyphs.
 *
 * Please note that this does not work with complex script such as Thai. They
 * need additional shaping.
 */
void cairo_font_draw_nonascii_centered(cairo_t *cr, double width, double height)
{
    cairo_font_face_t *droid;
    const char *text = NULL;
    cairo_text_extents_t extents;

    /* Create font that contains cyrillc glyphs */
    droid = cairo_font_from_ttf_file("/usr/share/fonts/ttf/DroidSans.ttf");

    /* Show simple rendering of non-latin text */
    cairo_set_font_face(cr, droid);
    cairo_set_font_size(cr, 50.0);

    /* Center it on the screen horizontally */
    /* Text means "Download finished */
    text = "\xd0\x97\xd0\xb0\xd0\xb2\xd0\xb5\xd1\x80\xd1\x88\xd0\xb5\xd0\xbd\xd0\xb8\xd1\x8f \xd0\xb7\xd0\xb0\xd0\xb3\xd1\x80\xd1\x83\xd0\xb7\xd0\xba\xd0\xb8";
    cairo_text_extents(cr, text, &extents);
    cairo_move_to(cr,
                  (width - extents.width) / 2.0,
                  (height - extents.height) / 2.0);
    cairo_show_text(cr, text);

    cairo_font_face_destroy(droid);
}
